<?php
defined( 'ABSPATH' ) || exit;

class HRM_Notifications {

    public static function init() {
         add_action( 'wp_ajax_hrm_mark_notification_read', [ __CLASS__, 'handle_mark_notification_read' ] );
         add_action( 'wp_ajax_hrm_mark_all_notifications_read', [ __CLASS__, 'handle_mark_all_notifications_read' ] );
         add_action( 'wp_ajax_hrm_check_notifications', [ __CLASS__, 'handle_check_notifications' ] );
         add_action( 'wp_ajax_hrm_get_notifications_page', [ __CLASS__, 'handle_get_notifications_page' ] );
    }

    /**
     * Get current employee ID for the logged-in user
     */
    private static function get_current_employee_id() {
        global $wpdb;
        $current_user_id = get_current_user_id();
        return $wpdb->get_var( $wpdb->prepare( 
            "SELECT ID FROM {$wpdb->posts} WHERE post_author = %d AND post_type = %s", 
            $current_user_id, HRM_Setup::EMPLOYEE_POST_TYPE 
        ));
    }

    /**
     * Get unread count for an employee
     */
    public static function get_unread_count( $employee_id ) {
        global $wpdb;
        return (int) $wpdb->get_var( $wpdb->prepare( 
            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_notifications WHERE employee_id = %d AND is_read = 0", 
            $employee_id 
        ));
    }

    public static function handle_mark_notification_read() {
        if ( ! is_user_logged_in() ) wp_send_json_error( 'Unauthorized' );
    
        $id = intval( $_POST['id'] );
        $employee_id = self::get_current_employee_id();
    
        if ( ! $employee_id ) wp_send_json_error( 'Employee not found' );
    
        global $wpdb;
        $updated = $wpdb->query( $wpdb->prepare( 
            "UPDATE {$wpdb->prefix}hrm_notifications SET is_read = 1 WHERE id = %d AND employee_id = %d", 
            $id, $employee_id 
        ));
    
        if ( $updated !== false ) {
            wp_send_json_success([
                'unread_count' => self::get_unread_count( $employee_id )
            ]);
        }
        wp_send_json_error( 'Failed to update' );
    }

    public static function handle_mark_all_notifications_read() {
        if ( ! is_user_logged_in() ) wp_send_json_error( 'Unauthorized' );
    
        $employee_id = self::get_current_employee_id();
    
        if ( ! $employee_id ) wp_send_json_error( 'Employee not found' );
    
        global $wpdb;
        $updated = $wpdb->query( $wpdb->prepare( 
            "UPDATE {$wpdb->prefix}hrm_notifications SET is_read = 1 WHERE employee_id = %d AND is_read = 0", 
            $employee_id 
        ));
    
        wp_send_json_success([
            'updated' => $updated,
            'unread_count' => 0
        ]);
    }

    public static function handle_check_notifications() {
        if ( ! is_user_logged_in() ) wp_send_json_error( 'Unauthorized' );
    
        $employee_id = self::get_current_employee_id();
    
        if ( ! $employee_id ) wp_send_json_error( 'Employee not found' );
    
        global $wpdb;
        $notifications = $wpdb->get_results( $wpdb->prepare( 
            "SELECT * FROM {$wpdb->prefix}hrm_notifications WHERE employee_id = %d AND is_read = 0 ORDER BY created_at DESC", 
            $employee_id 
        ));
    
        $data = [];
        foreach ( $notifications as $notif ) {
            $data[] = [
                'id' => $notif->id,
                'type' => $notif->type ?? 'general',
                'message' => $notif->message,
                'date' => date( 'M d, h:i A', strtotime( $notif->created_at ) )
            ];
        }
    
        wp_send_json_success([
            'notifications' => $data,
            'unread_count' => count($data)
        ]);
    }

    public static function create_notification( $recipient_id, $type, $message, $related_id = null ) {
        global $wpdb;
        $result = $wpdb->insert( $wpdb->prefix . 'hrm_notifications', [
            'employee_id' => $recipient_id,
            'type'        => $type,
            'message'     => $message,
            'related_id'  => $related_id,
            'is_read'     => 0,
            'created_at'  => current_time( 'mysql' )
        ]);
        
        return $result;
    }

    /**
     * Get notification type icon and color
     */
    private static function get_notification_icon( $type ) {
        $icons = [
            'task'     => ['icon' => 'dashicons-clipboard', 'color' => '#6366F1', 'bg' => '#EEF2FF'],
            'leave'    => ['icon' => 'dashicons-calendar-alt', 'color' => '#10B981', 'bg' => '#ECFDF5'],
            'approved' => ['icon' => 'dashicons-yes-alt', 'color' => '#10B981', 'bg' => '#ECFDF5'],
            'rejected' => ['icon' => 'dashicons-dismiss', 'color' => '#EF4444', 'bg' => '#FEF2F2'],
            'alert'    => ['icon' => 'dashicons-warning', 'color' => '#F59E0B', 'bg' => '#FFFBEB'],
            'message'  => ['icon' => 'dashicons-email', 'color' => '#3B82F6', 'bg' => '#EFF6FF'],
            'general'  => ['icon' => 'dashicons-info', 'color' => '#6B7280', 'bg' => '#F3F4F6'],
        ];
        
        return $icons[$type] ?? $icons['general'];
    }

    /**
     * Render the full notifications page for the employee portal
     */
    public static function render_notifications_page( $employee_id ) {
        global $wpdb;
        
        // Get pagination
        $per_page = isset($_GET['per_page']) ? max(5, min(100, intval($_GET['per_page']))) : 15;
        $current_page = isset($_GET['notif_page']) ? max(1, intval($_GET['notif_page'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        
        // Get total count
        $total = $wpdb->get_var( $wpdb->prepare( 
            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_notifications WHERE employee_id = %d", 
            $employee_id 
        ));
        
        // Get notifications
        $notifications = $wpdb->get_results( $wpdb->prepare( 
            "SELECT * FROM {$wpdb->prefix}hrm_notifications WHERE employee_id = %d ORDER BY is_read ASC, created_at DESC LIMIT %d OFFSET %d", 
            $employee_id, $per_page, $offset 
        ));
        
        $unread_count = self::get_unread_count( $employee_id );
        ?>
        <div class="hrm-notifications-page">
            <!-- Page Header -->
            <div class="hrm-page-header">
                <div class="header-left">
                    <h2><span class="dashicons dashicons-bell"></span> Notifications</h2>
                    <p>Stay updated with your latest alerts and messages</p>
                </div>
                <div class="header-right">
                    <?php if ( $unread_count > 0 ) : ?>
                        <button type="button" class="hrm-btn-outline btn-sm" id="mark-all-read-btn">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <span class="btn-text">Mark All as Read</span>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Stats Bar -->
            <div class="hrm-notif-stats">
                <div class="notif-stat">
                    <span class="stat-value"><?php echo $total; ?></span>
                    <span class="stat-label">Total</span>
                </div>
                <div class="notif-stat unread">
                    <span class="stat-value" id="unread-count-display"><?php echo $unread_count; ?></span>
                    <span class="stat-label">Unread</span>
                </div>
                <div class="notif-stat read">
                    <span class="stat-value"><?php echo $total - $unread_count; ?></span>
                    <span class="stat-label">Read</span>
                </div>
            </div>
            
            <!-- Notifications List -->
            <div class="hrm-notifications-container">
                <?php if ( empty($notifications) ) : ?>
                    <div class="hrm-empty-state">
                        <div class="empty-icon">
                            <span class="dashicons dashicons-bell"></span>
                        </div>
                        <h3>No Notifications</h3>
                        <p>You're all caught up! New notifications will appear here.</p>
                    </div>
                <?php else : ?>
                    <ul class="hrm-notifications-list" id="notifications-list">
                        <?php foreach ( $notifications as $notif ) : 
                            $icon_data = self::get_notification_icon( $notif->type ?? 'general' );
                            $is_unread = !$notif->is_read;
                            $time_ago = human_time_diff( strtotime($notif->created_at), current_time('timestamp') ) . ' ago';
                        ?>
                            <li class="notification-item <?php echo $is_unread ? 'unread' : 'read'; ?>" data-id="<?php echo $notif->id; ?>">
                                <div class="notif-icon" style="background: <?php echo $icon_data['bg']; ?>; color: <?php echo $icon_data['color']; ?>;">
                                    <span class="dashicons <?php echo $icon_data['icon']; ?>"></span>
                                </div>
                                <div class="notif-content">
                                    <p class="notif-message"><?php echo esc_html( $notif->message ); ?></p>
                                    <div class="notif-meta">
                                        <span class="notif-time">
                                            <span class="dashicons dashicons-clock"></span>
                                            <?php echo esc_html( $time_ago ); ?>
                                        </span>
                                        <span class="notif-date"><?php echo date( 'M j, Y \a\t g:i A', strtotime( $notif->created_at ) ); ?></span>
                                    </div>
                                </div>
                                <div class="notif-actions">
                                    <?php if ( $is_unread ) : ?>
                                        <button type="button" class="mark-read-btn" data-id="<?php echo $notif->id; ?>" title="Mark as read">
                                            <span class="dashicons dashicons-yes"></span>
                                        </button>
                                    <?php else : ?>
                                        <span class="read-indicator" title="Read">
                                            <span class="dashicons dashicons-saved"></span>
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    
                    <!-- Pagination -->
                    <?php 
                    HRM_Pagination::render([
                        'total_items' => $total,
                        'current_page' => $current_page,
                        'per_page' => $per_page,
                        'param_name' => 'notif_page',
                        'show_sizes' => true,
                        'context' => 'frontend'
                    ]);
                    HRM_Pagination::render_script();
                    ?>
                <?php endif; ?>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Mark single notification as read
            $(document).on('click', '.mark-read-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var id = btn.data('id');
                var item = btn.closest('.notification-item');
                
                btn.prop('disabled', true);
                
                $.post(hrm_ajax.ajaxurl, {
                    action: 'hrm_mark_notification_read',
                    id: id
                }, function(res) {
                    if(res.success) {
                        item.removeClass('unread').addClass('read');
                        btn.replaceWith('<span class="read-indicator" title="Read"><span class="dashicons dashicons-saved"></span></span>');
                        updateUnreadCount(res.data.unread_count);
                    }
                });
            });
            
            // Mark all as read
            $('#mark-all-read-btn').on('click', function() {
                var btn = $(this);
                btn.prop('disabled', true).find('.btn-text').text('Marking...');
                
                $.post(hrm_ajax.ajaxurl, {
                    action: 'hrm_mark_all_notifications_read'
                }, function(res) {
                    if(res.success) {
                        $('.notification-item.unread').each(function() {
                            $(this).removeClass('unread').addClass('read');
                            $(this).find('.mark-read-btn').replaceWith('<span class="read-indicator" title="Read"><span class="dashicons dashicons-saved"></span></span>');
                        });
                        updateUnreadCount(0);
                        btn.fadeOut();
                    }
                });
            });
            
            // Update unread count in all badges
            function updateUnreadCount(count) {
                $('#unread-count-display').text(count);
                if (count > 0) {
                    $('.notif-badge, .nav-notif-badge').text(count).show();
                } else {
                    $('.notif-badge, .nav-notif-badge').hide();
                }
            }
        });
        
        // Load notifications page (for pagination)
        function loadNotificationsPage(page) {
            // Simple page reload with query param
            var url = new URL(window.location.href);
            url.searchParams.set('notif_page', page);
            window.location.href = url.toString();
        }
        </script>
        <?php
    }

    public static function render_frontend_notifications( $employee_id ) {
        $unread_count = self::get_unread_count( $employee_id );
        // We output the container for toasts
        ?>
        <div id="hrm-toast-container" class="hrm-toast-container" style="position: fixed; bottom: 20px; right: 20px; z-index: 9999; display: flex; flex-direction: column; gap: 10px; max-width: 350px;"></div>
        
        <script>
        jQuery(document).ready(function($) {
            let knownNotifs = [];
            let initialUnread = <?php echo $unread_count; ?>;

            // Helper to update badge
            function updateBadge(count) {
                const badge = $('.notif-badge');
                const navBadge = $('.nav-notif-badge');
                if (count > 0) {
                    badge.text(count).css('display', 'flex');
                    navBadge.text(count).show();
                    $('.hrm-notif-bell').addClass('has-notifs');
                } else {
                    badge.hide();
                    navBadge.hide();
                    $('.hrm-notif-bell').removeClass('has-notifs');
                }
            }

            // Initialize badge
            updateBadge(initialUnread);

            // Helper to show toast
            function showToast(id, message, date) {
                const html = `
                    <div id="hrm-notif-${id}" class="hrm-toast" style="background: #fff; padding: 15px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); display: flex; align-items: flex-start; gap: 10px; border-left: 4px solid var(--primary-color); opacity: 0; transform: translateY(20px); transition: all 0.3s;">
                        <div class="toast-content" style="flex: 1;">
                            <p style="margin: 0; font-size: 14px; color: #374151;">${message}</p>
                            <small style="color: #9CA3AF; font-size: 11px;">${date}</small>
                        </div>
                        <button class="hrm-close-toast" data-id="${id}" style="background: none; border: none; color: #9CA3AF; cursor: pointer; font-size: 18px; line-height: 1;">&times;</button>
                    </div>
                `;
                const $toast = $(html);
                $('#hrm-toast-container').append($toast);
                
                // Animate in
                setTimeout(() => $toast.css({opacity: 1, transform: 'translateY(0)'}), 10);

                // Auto hide after 5s
                setTimeout(function() {
                    $toast.css({opacity: 0, transform: 'translateY(20px)'});
                    setTimeout(() => $toast.remove(), 300);
                }, 6000);
            }

            function checkNotifications() {
                if (typeof hrm_ajax === 'undefined') return;
                $.post(hrm_ajax.ajaxurl, {
                    action: 'hrm_check_notifications',
                    nonce: hrm_ajax.nonce
                }, function(response) {
                    if(response.success) {
                        const notifs = response.data.notifications;
                        updateBadge(response.data.unread_count);

                        notifs.forEach(function(n) {
                            if (!knownNotifs.includes(n.id)) {
                                knownNotifs.push(n.id);
                                showToast(n.id, n.message, n.date);
                            }
                        });
                    }
                });
            }

            // Initial Check
            checkNotifications();

            // Poll every 30s
            setInterval(checkNotifications, 30000);

            // Handle Toast Close / Mark Read
            $(document).on('click', '.hrm-close-toast', function() {
                const id = $(this).data('id');
                // Optimistically remove toast
                $(this).closest('.hrm-toast').remove();
                
                // Mark read on server
                if (typeof hrm_ajax === 'undefined') return;
                $.post(hrm_ajax.ajaxurl, {
                    action: 'hrm_mark_notification_read',
                    id: id
                }, function(res) {
                    if(res.success) {
                       checkNotifications();
                    }
                });
            });
            
            // Bell Click - Navigate to notifications tab
            $('.hrm-notif-bell').on('click', function() {
                 // Find and click the notifications tab
                 var notifTab = $('.hrm-tab-btn[onclick*="notifications"]');
                 if (notifTab.length) {
                     notifTab.click();
                 }
            });
        });
        </script>
        <?php
    }
    
    public static function render_frontend_notifications_list( $employee_id ) {
        global $wpdb;
        $notifications = $wpdb->get_results( $wpdb->prepare( 
            "SELECT * FROM {$wpdb->prefix}hrm_notifications WHERE employee_id = %d ORDER BY created_at DESC LIMIT 5", 
            $employee_id 
        ));
        $unread_count = self::get_unread_count( $employee_id );
        ?>
        <div class="hrm-notifications-list-box">
            <div class="notif-box-header">
                <h3><span class="dashicons dashicons-bell"></span> Recent Notifications</h3>
                <?php if ( $unread_count > 0 ) : ?>
                    <span class="unread-badge"><?php echo $unread_count; ?> unread</span>
                <?php endif; ?>
            </div>
            <?php if ( $notifications ) : ?>
                <ul class="hrm-notif-list">
                    <?php foreach ( $notifications as $notif ) : ?>
                        <li class="<?php echo $notif->is_read ? 'read' : 'unread'; ?>">
                            <p><?php echo esc_html( $notif->message ); ?></p>
                            <small><?php echo date( 'M j, Y g:i a', strtotime( $notif->created_at ) ); ?></small>
                        </li>
                    <?php endforeach; ?>
                </ul>
                <div class="notif-box-footer">
                    <button type="button" class="hrm-btn hrm-btn-primary hrm-btn-view-all-notifs" onclick="openTab(event, 'notifications')">
                        View All Notifications →
                    </button>
                </div>
            <?php else : ?>
                <p class="no-notifs">No notifications yet.</p>
            <?php endif; ?>
        </div>
        <?php
    }
}

